using System.Collections.Generic;
using Newtonsoft.Json;
using UnityEngine;


[System.Serializable]
public class InitResult
{
    public string channel;
    public int gameId;
    public int channelId;
}

[System.Serializable]
public class LoginResult
{
    public long uid;
    public string accessToken;
    public string channel;
    public string channelUid;   // 已弃用,默认为0 
    public string serverArea;
    public string extendInfo;   // 扩展参数转 JSON 字符串再传过来
}

[System.Serializable]
public class GoodsInfo
{
    public string currency;
    public string itemId;
    public string itemName;
    public string itemCount;
    public string itemPrice;
    public string showTag;
    public string region;//仅 iOS返回
    public string symbol;//仅 iOS返回
}

public class PlatformCallback : MonoBehaviour
{
    private static IPlatformCallbackListener _listener;
    private string SDK_BUILD_VERSION = "1.1.0.0";
    private string SDK_20251023 = "202510231755";

    // 单例实现，确保只有一个接收器实例
    public static PlatformCallback Instance { get; private set; }

    void Awake()
    {
        Debug.Log($"[PlatformCallback] unity sdk version:{SDK_BUILD_VERSION} Build Time:{SDK_20251023}");
        // 单例模式实现
        if (Instance != null && Instance != this)
        {
            NativeLogger.I("PlatformCallback", "[PlatformCallback] Destroying duplicate instance");
            Destroy(gameObject);
            return;
        }

        Instance = this;

        // 确保接收器在场景切换中不被销毁
        DontDestroyOnLoad(gameObject);

        // 确保游戏对象名称与 Android 调用匹配
        gameObject.name = "PlatformCallback";

        NativeLogger.I("PlatformCallback", $"[PlatformCallback] Initialized with GameObject name: {gameObject.name}");
    }

    public static void SetListener(IPlatformCallbackListener listener)
    {
        _listener = listener;
    }

    public void onInitSuccess(string jsonStr)
    {
        NativeLogger.I("PlatformCallback", "[PlatformCallback] onInitSuccess called with result: " + jsonStr);
        InitResult result = JsonConvert.DeserializeObject<InitResult>(jsonStr);
        if (result != null)
        {
            _listener?.OnInitSuccess(result);
        }
    }

    public void onInitFailed(string reason)
    {
        NativeLogger.I("PlatformCallback", "[PlatformCallback] onInitFailed called with reason: " + reason);
        _listener?.OnInitFailed(reason);
    }

    //自定义退出界面
    public void onCustomExit()
    {
        NativeLogger.I("PlatformCallback", "[PlatformCallback] onCustomExit called");
        _listener?.OnCustomExit();
    }

    //预留
    public void onUpdate(string data)
    {
        NativeLogger.I("PlatformCallback", "[PlatformCallback] onUpdate called with data: " + data);
        _listener?.OnUpdate(data);
    }

    //登录成功
    public void onLoginSuccess(string jsonStr)
    {
        NativeLogger.I("PlatformCallback", "[PlatformCallback] onLoginSuccess called with result: " + jsonStr);


        //判断 null 或空字符串判断空白字符串（包括空格、制表符等）
        if (string.IsNullOrEmpty(jsonStr) || string.IsNullOrWhiteSpace(jsonStr))
        {
            NativeLogger.I("PlatformCallback", "[Unity] Android returned empty or invalid string");
            return;
        }
        LoginResult loginResult = JsonConvert.DeserializeObject<LoginResult>(jsonStr);

        if (loginResult.uid != 0)
        {
            _listener?.OnLoginSuccess(loginResult);
        }

    }

    public void onRefreshUser(string jsonStr)
    {
        NativeLogger.I("PlatformCallback", "[PlatformCallback] onRefreshUser called with result: " + jsonStr);

        //判断 null 或空字符串判断空白字符串（包括空格、制表符等）
        if (string.IsNullOrEmpty(jsonStr) || string.IsNullOrWhiteSpace(jsonStr))
        {
            NativeLogger.I("PlatformCallback", "[PlatformCallback] Android returned empty or invalid string");
            return;
        }
        LoginResult loginResult = JsonConvert.DeserializeObject<LoginResult>(jsonStr);

        if (loginResult.uid != 0)
        {
            _listener?.OnRefreshUser(loginResult);
        }
    }

    public void onLoginFailed(string reason)
    {
        NativeLogger.I("PlatformCallback", "[PlatformCallback] onLoginFailed called with reason: " + reason);
        _listener?.OnLoginFailed(reason);
    }

    public void onLogout(string message)
    {
        NativeLogger.I("PlatformCallback", "[PlatformCallback] onLogout called with message: " + message);
        _listener?.OnLogout(message);
    }

    public void onPayGoodsList(string jsonStr)
    {
        NativeLogger.I("PlatformCallback", "[PlatformCallback] onPayGoodsList called with message: " + jsonStr);
        List<GoodsInfo> goodsLists = JsonConvert.DeserializeObject<List<GoodsInfo>>(jsonStr);
        if (goodsLists != null && goodsLists.Count != 0)
        {
            _listener?.OnPayGoodsList(goodsLists);
        }
    }

    public void onPaySuccess(string jsonStr)
    {
        NativeLogger.I("PlatformCallback", $"[PlatformCallback] onPaySuccess called {jsonStr}");
        _listener?.OnPaySuccess(jsonStr);
    }

    public void onPayFailed(string reason)
    {
        NativeLogger.I("PlatformCallback", "[PlatformCallback] onPayFailed called with reason: " + reason);
        _listener?.OnPayFailed(reason);
    }


    //分享，无特殊处理可不实现
    public void onShareSuccess(string jsonStr)
    {
        NativeLogger.I("PlatformCallback", $"[PlatformCallback] onShareSuccess called {jsonStr}");
        _listener?.OnShareSuccess(jsonStr);
    }

    public void onShareFailed(string jsonStr)
    {
        NativeLogger.I("PlatformCallback", $"[PlatformCallback] onShareFailed called {jsonStr}");
        _listener?.OnShareFailed(jsonStr);
    }

}
