using UnityEngine;

/// <summary>
/// 原生日志记录器。
/// 该类直接调用 Android 或 iOS 的系统级日志API，可以绕过 Unity 的 Debug.Log 剥离机制。
/// 这意味着即使在 Release Build (非 Development Build) 中，日志也能被输出到 Logcat (Android) 或 Console (iOS)。
/// </summary>
public static class NativeLogger
{
#if UNITY_ANDROID && !UNITY_EDITOR
    // 为 Android 平台缓存 Java Log Class，避免重复查找
    private static readonly AndroidJavaClass _logClass = new AndroidJavaClass("android.util.Log");
#endif

    /// <summary>
    /// 打印一条信息级别的日志。
    /// </summary>
    /// <param name="tag">Logcat 中显示的标签</param>
    /// <param name="message">日志内容</param>
    public static void I(string tag, object message)
    {
        Debug.Log($"[{tag}] {message}");
#if UNITY_EDITOR
        // 在编辑器中，我们仍然使用 Debug.Log，因为它功能更丰富（例如可以点击跳转）
        Debug.Log($"[{tag}] {message}");
#elif UNITY_ANDROID
        // 在 Android 设备上，直接调用 android.util.Log.i(tag, message)
        _logClass.CallStatic<int>("i", tag, message.ToString());
#elif UNITY_IOS
        // 在 iOS 设备上，调用自定义的 NSLog 封装
        // (需要在 Xcode 项目中添加一个 .mm 文件来实现 _logToConsole)
         //_logToConsole($"[I] [{tag}] {message}");
#endif
    }

    /// <summary>
    /// 打印一条警告级别的日志。
    /// </summary>
    /// <param name="tag">Logcat 中显示的标签</param>
    /// <param name="message">日志内容</param>
    public static void W(string tag, object message)
    {
#if UNITY_EDITOR
        Debug.LogWarning($"[{tag}] {message}");
#elif UNITY_ANDROID
        _logClass.CallStatic<int>("w", tag, message.ToString());
#elif UNITY_IOS
        // _logToConsole($"[W] [{tag}] {message}");
#endif
    }

    /// <summary>
    /// 打印一条错误级别的日志。
    /// </summary>
    /// <param name="tag">Logcat 中显示的标签</param>
    /// <param name="message">日志内容</param>
    public static void E(string tag, object message)
    {
#if UNITY_EDITOR
        Debug.LogError($"[{tag}] {message}");
#elif UNITY_ANDROID
        _logClass.CallStatic<int>("e", tag, message.ToString());
#elif UNITY_IOS
        // _logToConsole($"[E] [{tag}] {message}");
#endif
    }
}
