﻿using UnityEngine;
using System.Collections;
using System.IO;

public class NativeShareHelper
{
    private const string SCREENSHOT_FILENAME = "screenshot.png";

    /// <summary>
    /// 截取当前游戏屏幕，保存为图片，并返回其本地路径。
    /// 这是一个异步操作，因为它需要等待渲染帧的结束。
    /// </summary>
    /// <param name="onScreenshotSaved">截图完成后的回调，参数为图片的本地绝对路径。</param>
    /// <returns>返回一个 Coroutine，需要由 MonoBehaviour 的实例来启动。</returns>
    public static IEnumerator CaptureAndSaveScreenshot(System.Action<string> onScreenshotSaved)
    {
        // 关键步骤：必须等待渲染帧的末尾，才能确保截取到完整的、最新的屏幕画面。
        yield return new WaitForEndOfFrame();

        // 1. 创建一个与屏幕同等大小的 Texture2D 用于接收截图数据
        Texture2D screenshotTexture = new Texture2D(Screen.width, Screen.height, TextureFormat.RGB24, false);

        // 2. 读取屏幕像素到 Texture2D
        Rect screenArea = new Rect(0, 0, Screen.width, Screen.height);
        screenshotTexture.ReadPixels(screenArea, 0, 0);
        screenshotTexture.Apply();

        // 3. 将 Texture2D 保存为文件，并获取路径
        string filePath = SaveTextureToFile(screenshotTexture, SCREENSHOT_FILENAME);

        // 4. 销毁临时的 Texture2D 对象，防止内存泄漏
        Object.Destroy(screenshotTexture);

        // 5. 通过回调函数返回文件路径
        onScreenshotSaved?.Invoke(filePath);
    }

    /// <summary>
    /// 将一个已有的 Texture2D 资源保存为本地文件，并返回其路径。
    /// 这是一个同步操作，但请注意 EncodeToPNG() 可能有性能开销。
    /// </summary>
    /// <param name="texture">要保存的图片资源。重要：此 Texture 必须在 Inspector 中开启 "Read/Write Enabled" 选项。</param>
    /// <param name="fileName">指定保存的文件名，例如 "my_downloaded_image.png"。</param>
    /// <returns>返回图片的本地绝对路径。如果保存失败，则返回 null。</returns>
    public static string SaveTextureToFile(Texture2D texture, string fileName)
    {
        if (texture == null)
        {
            Debug.LogError("[NativeShareHelper] 传入的 Texture2D 为空！");
            return null;
        }

        try
        {
            // 1. 将 Texture2D 编码为 PNG 格式的字节数组
            byte[] bytes = texture.EncodeToPNG();

            // 2. 构造目标文件的完整路径 (核心)
            // Path.Combine 会自动处理不同操作系统的路径分隔符 ('/' 或 '\')，非常安全可靠。
            string filePath = Path.Combine(Application.persistentDataPath, fileName);

            // 3. 将字节数组写入文件。如果文件已存在，此方法会覆盖它。
            File.WriteAllBytes(filePath, bytes);

            Debug.Log($"[NativeShareHelper] 图片已成功保存至: {filePath}");

            // 4. 返回这个可供原生层使用的绝对路径
            return filePath;
        }
        catch (UnityException ex)
        {
            // 这是最常见的错误，因为 Texture 默认是不可读的 GPU 资源。
            Debug.LogError($"[NativeShareHelper] 保存图片失败！请确保 Texture '{texture.name}' 在 Inspector 中已勾选 'Read/Write Enabled' 选项。错误详情: {ex.Message}");
            return null;
        }
        catch (System.Exception ex)
        {
            Debug.LogError($"[NativeShareHelper] 写入文件时发生未知 IO 错误: {ex.Message}");
            return null;
        }
    }
}
