using UnityEngine;
using UnityEngine.Scripting;

/// <summary>
/// 原生日志记录器。
/// 该类直接调用 Android 或 iOS 的系统级日志API，可以绕过 Unity 的 Debug.Log 剥离机制。
/// 这意味着即使在 Release Build (非 Development Build) 中，日志也能被输出到 Logcat (Android) 或 Console (iOS)。
/// </summary>
[Preserve]
public static class NativeLogger
{
#if UNITY_ANDROID && !UNITY_EDITOR
    private static readonly AndroidJavaClass _logClass = new AndroidJavaClass("android.util.Log");
#endif

    /// <summary>
    /// 打印一条信息级别的日志。
    /// </summary>
    /// <param name="tag">Logcat 中显示的标签</param>
    /// <param name="message">日志内容</param>
    public static void I(string tag, object message)
    {
        Debug.Log($"[{tag}] {message}");
#if UNITY_ANDROID && !UNITY_EDITOR
        // 在 Android 设备上，额外调用 android.util.Log.i 输出到 Logcat
        _logClass.CallStatic<int>("i", tag, message.ToString());
#endif
    }

    /// <summary>
    /// 打印一条警告级别的日志。
    /// </summary>
    /// <param name="tag">Logcat 中显示的标签</param>
    /// <param name="message">日志内容</param>
    public static void W(string tag, object message)
    {
        Debug.LogWarning($"[{tag}] {message}");
#if UNITY_ANDROID && !UNITY_EDITOR
        _logClass.CallStatic<int>("w", tag, message.ToString());
#endif
    }

    /// <summary>
    /// 打印一条错误级别的日志。
    /// </summary>
    /// <param name="tag">Logcat 中显示的标签</param>
    /// <param name="message">日志内容</param>
    public static void E(string tag, object message)
    {
        Debug.LogError($"[{tag}] {message}");
#if UNITY_ANDROID && !UNITY_EDITOR
        _logClass.CallStatic<int>("e", tag, message.ToString());
#endif
    }
}
